/**
* \file: PfCfgConfiguration.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <fstream>
#include <algorithm>
#include "Common.h"
#include "PfCfgConfiguration.h"
#include "utils/Statistics.h"

namespace adit { namespace carplay
{

using namespace std;

PfCfgConfiguration* PfCfgConfiguration::FromFile(const string& file)
{
    ifstream cfg(file);
    if (cfg.fail())
        return nullptr;

    auto result = new PfCfgConfiguration();

    string line;
    while (getline(cfg, line))
    {
        const char* spaces = " \t";

        // check for # comments
        size_t comment = line.find('#');
        if (comment != string::npos)
        {
            line = line.substr(0, comment);
        }

        // find first non-space character
        auto keyBegin = line.find_first_not_of(spaces);
        // line with non-space content
        if (keyBegin != string::npos)
        {
            // find first space character
            auto keyEnd = line.find_first_of(spaces, keyBegin);
            // continue only if
            if (keyEnd != string::npos)
            {
                string key = line.substr(keyBegin, keyEnd - keyBegin);

                // find first non-space character
                auto valueBegin = line.find_first_not_of(spaces, keyEnd);
                string value;

                // value found
                if (valueBegin != string::npos)
                {
                    // remove trailing spaces
                    size_t lastNonSpace = line.find_last_not_of(spaces);
                    value = line.substr(valueBegin, lastNonSpace - valueBegin + 1);
                }
                else
                {
                    // set to empty string
                    value = "";
                }

                result->insert(key, value);
            }
        }
    }

    return result;
}

PfCfgConfiguration::PfCfgConfiguration()
{
}

PfCfgConfiguration::~PfCfgConfiguration()
{
}

void PfCfgConfiguration::SetItem(const std::string& key, const std::string& value)
{
    config[key] = value;
}

bool PfCfgConfiguration::TryGetItem(const string& key, string& result) const
{
    string value;
    // check if item is available
    if (checkItem(key, value))
    {
        LOGD_DEBUG((dipo, "config item %s = \"%s\"", key.c_str(), value.c_str()));
        result = value;
        return true;
    }

    // item was not found
    result = "";
    return false;
}

list<string> PfCfgConfiguration::GetItems(const string& key) const
{
    const int keyLength = key.length() + 10;
    list<string> list;

    for (int i = 0; true; i++)
    {
        char singleKey[keyLength];
        snprintf(singleKey, keyLength, "%s-%d", key.c_str(), i);

        auto found = config.find(singleKey);

        // add found item to list
        if (found != config.end())
        {
            LOGD_DEBUG((dipo, "config item %s = \"%s\"", key.c_str(),
                    found->second.c_str()));
            list.push_back(found->second);
        }
        else
        {
            // in case no item was found with -xx format, fallback to single line
            if (i == 0)
            {
                found = config.find(key);
                if (found != config.end())
                {
                    LOGD_DEBUG((dipo, "config item %s = \"%s\"", key.c_str(),
                            found->second.c_str()));
                    list.push_back(found->second);
                }
            }
            break;
        }
    }

    return list;
}

list<string> PfCfgConfiguration::GetStatistics()
{
    list<string> stats;

    for (auto it : config)
    {
        string result = it.first + " = " + it.second;
        stats.push_back(result);
    }

    return stats;
}

bool PfCfgConfiguration::checkItem(const string& key, string& value) const
{
    // check if item is available
    auto found = config.find(key);
    if (found != config.end())
    {
        value = found->second;
        return true;
    }

    return false;
}

void PfCfgConfiguration::insert(const string& inKey, const string& inValue)
{
    static int lastIndex = 1;
    static string lastKey;

    if (config.find(inKey) == config.end())
    {
        config[inKey] = inValue;
    }
    else
    {
        string newKey = inKey + "-0";
        if (config.find(newKey) == config.end())
        {
            config[newKey] = config[inKey];
        }

        int i = 1;
        if (lastKey.compare(inKey) == 0)
            i = lastIndex;
        for (; i < 1000; i++)
        {
            char buf[1024];
            snprintf(buf, sizeof(buf), "%s-%d", inKey.c_str(), i);
            if (config.find(buf) == config.end())
            {
                config[buf] = inValue;
                lastIndex = i;
                lastKey = inKey;
                break;
            }
        }

        if (i == 1000)
        {
            LOG_ERROR((dipo, "too many configuration lines for %s", inKey.c_str()));
        }
    }
}

} } // namespace adit { namespace carplay
